// Rad_Move.js //
// Entity and player movement counterpart //

// Basic movement command. Will move person 'name', 'x' amount of
// Tiles, in a certain direction, facing a certain way. 
function Move(name, tiles, direction_command, face_command)
{
	if (face_command === undefined)
	{
		if (direction_command == COMMAND_MOVE_NORTH) face_command = COMMAND_FACE_NORTH;
		if (direction_command == COMMAND_MOVE_SOUTH) face_command = COMMAND_FACE_SOUTH;
		if (direction_command == COMMAND_MOVE_EAST) face_command = COMMAND_FACE_EAST;
		if (direction_command == COMMAND_MOVE_WEST) face_command = COMMAND_FACE_WEST;
		if (direction_command == COMMAND_WAIT) face_command = null;
	}
	
	if (!IsObstructed(name, direction_command))
	{
		if (face_command != null) QueuePersonCommand(name, face_command, true);
		for (var i = 0; i < tiles; ++i) MovePixels(name, direction_command, GetTileWidth());
	}
}

// This is used to check if a given person can move
// towards a given direction.
function IsObstructed(name, direction)
{
	const X = GetPersonX(name);
	const Y = GetPersonY(name);
	switch(direction)
	{
		case COMMAND_MOVE_NORTH:
			if (IsPersonObstructed(name, X, Y-1)) return true;
		break;
		case COMMAND_MOVE_SOUTH:
			if (IsPersonObstructed(name, X, Y+1)) return true;
		break;
		case COMMAND_MOVE_EAST:
			if (IsPersonObstructed(name, X+1, Y)) return true;
		break;
		case COMMAND_MOVE_WEST:
			if (IsPersonObstructed(name, X-1, Y)) return true;
		break;
		default: return false; break;
	}
}

// Utilize the pathfinding algorithm to move
// An entity towards a given destination.
function MoveToLocation(name, x, y, wait)
{
	var Path = FindPath(name, -1, -1, 1, x, y);
	MovePath(name, Path, wait);
}

function MovePath(name, path, wait)
{
	if (wait == undefined) wait = false;
	for (var i = 0; i < path.length; ++i)
	{
		switch (path.charAt(i))
		{
			case "E": Move(name, 1, EAST); break;
			case "N": Move(name, 1, NORTH); break;
			case "S": Move(name, 1, SOUTH); break;
			case "W": Move(name, 1, WEST); break;
			case "F": break;
		}
	}
	if (wait) WaitForPerson(name);
}

function ReversePath(path)
{
	var temp_path = "";
	for (var i = path.length; i > -1; --i)
	{
		switch (path.charAt(i))
		{
			case "E": temp_path += "W"; break;
			case "N": temp_path += "S"; break;
			case "S": temp_path += "N"; break;
			case "W": temp_path += "E"; break;
		}
	}
	return temp_path;
}

// Blocks other things from happening, but
// It will update anything that would be
// Updateable in a given timeframe.
// EX: Player movement.
function WaitForPerson(msecs)
{
	var time = GetTime();
	var framerate = GetFrameRate();
	SetFrameRate(GetMapEngineFrameRate());
	while (time + msecs > GetTime())
	{
		UpdateMapEngine();
		RenderMap();
		FlipScreen();
	}
	SetFrameRate(framerate);
}

function MovePixels(name, direction, amount)
{
	for (var i = 0; i < amount; ++i) QueuePersonCommand(name, direction, false);
}

function DoesPersonExist(name)
{
	var List = GetPersonList();
	for (var i = 0; i < List.length; ++i)
	{
		if (List[i] == name) return true;
	}
	return false;
}

function SetPersonToTile(name, x, y)
{
	SetPersonX(name, (x>>4)*GetTileWidth());
	SetPersonY(name, (y>>4)*GetTileHeight());
}

function SetPersonXY(name, x, y)
{
	SetPersonX(name, x*16+7);
	SetPersonY(name, y*16+7);
}

// should be sparingly called //
// filter is used to resolve a 2-people at same position problem.
function GetPersonAt(x, y, filter, layer)
{
	var pList = GetPersonList();
	var i = pList.length;
	while(i--)
	{
		var name = pList[i];
		var base = GetPersonBase(name);
		var w = base.x2 - base.x1;
		var h = base.y2 - base.y1;
		var px = GetPersonX(name)-w/2;
		var py = GetPersonY(name)-h/2;
		var l = GetPersonLayer(name);
		if (filter) { if (name.indexOf(filter) == -1) continue; }
		if (layer) { if (l != layer) continue; }
		if (x > px && x < px+w+1 && y > py && y < py+h+1) return name;
	}
	return "";
}

function Wait(msecs)
{
	var time = GetTime();
	while (time + msecs > GetTime())
	{
		UpdateMapEngine();
		RenderMap();
		FlipScreen();
	}
}